const express = require('express');
const pool = require('../db');
const { authenticateToken } = require('./auth');

const router = express.Router();

// Share a task with another user by username or email
router.post('/', authenticateToken, async (req, res) => {
  const { task_id, shared_with_username } = req.body;
  if (!task_id || !shared_with_username) return res.status(400).json({ error: 'Task ID and username required' });

  try {
    // Check if task belongs to current user
    const [tasks] = await pool.query('SELECT * FROM tasks WHERE id = ? AND user_id = ?', [task_id, req.user.id]);
    if (tasks.length === 0) return res.status(404).json({ error: 'Task not found or unauthorized' });

    // Find user to share with
    const [users] = await pool.query('SELECT * FROM users WHERE username = ?', [shared_with_username]);
    if (users.length === 0) return res.status(404).json({ error: 'User to share with not found' });

    const sharedWithUser = users[0];

    // Check if already shared
    const [existing] = await pool.query('SELECT * FROM task_shares WHERE task_id = ? AND shared_with_user_id = ?', [task_id, sharedWithUser.id]);
    if (existing.length) return res.status(400).json({ error: 'Task already shared with this user' });

    await pool.query('INSERT INTO task_shares (task_id, shared_with_user_id, shared_by_user_id) VALUES (?, ?, ?)', [task_id, sharedWithUser.id, req.user.id]);

    res.json({ message: `Task shared with ${sharedWithUser.username}` });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// Get tasks shared with the current user
router.get('/shared-with-me', authenticateToken, async (req, res) => {
  try {
    const [tasks] = await pool.query(`
      SELECT t.*, c.name as category_name, u.username as owner_username 
      FROM task_shares ts
      JOIN tasks t ON ts.task_id = t.id
      JOIN categories c ON t.category_id = c.id
      JOIN users u ON t.user_id = u.id
      WHERE ts.shared_with_user_id = ?
      ORDER BY t.updated_at DESC
    `, [req.user.id]);
    res.json(tasks);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

module.exports = router;
